
import unittest
import wget
import threading
from BaseHTTPServer import BaseHTTPRequestHandler, HTTPServer
import cgi
import re
from BeautifulSoup import BeautifulSoup
import sys
import signal
import os


TEST_HOST = '127.0.0.1'
TEST_PORT = 12777

class MockServerHandler(BaseHTTPRequestHandler):

    def do_GET(self):
        """ handle GET request """
        self.send_response(200)
        self.send_header('Content-Type', 'text/html')
        self.send_header('Set-Cookie', 'cookieName=cookieValue; path=/')
        self.end_headers()

        self.wfile.write("<test>AAA&nbsp;BBB</test>\n")
        self.wfile.write("<method>%r</method>\n" % self.command)
        self.wfile.write("<path>%r</path>\n" % self.path.replace('&','&amp;'))
        for (key, val) in self.headers.items():
            self.wfile.write("<header name=%r>%r</header>\n" % (key, val))

    def do_POST(self):
        """ handle POST request """
        self.do_GET()
        form = cgi.FieldStorage(
                        fp=self.rfile,
                        headers=self.headers,
                        environ={'REQUEST_METHOD':'POST',
                                'CONTENT_TYPE':self.headers['Content-Type'],
                            })
        for key in form.keys():
            self.wfile.write("<data name=%r>%r</data>\n" \
                                % (key, form.getvalue(key)))


    def log_request(*ingnore):
        pass
        


class MockServer(HTTPServer):
    timeout = 0.1

    def __init__(self):
        """ Constructor """
        bind = (TEST_HOST, TEST_PORT)
        HTTPServer.__init__(self, bind, MockServerHandler)


    def run(self):
        """ Run http testing server in separate thread """
        self._quit = threading.Event()
        self._thread = threading.Thread(target = self.serve)
        self._thread.start()


    def serve(self):
        """ server main loop """
        while not self._quit.isSet():
            self.handle_request()
            self._quit.wait(0.005)


    def quit(self):
        """ Exit runnig server """
        self._quit.set()
        self._thread.join(5)
        if self._thread.isAlive():
            sys.stderr.write("Cannot terminate MockServer thread - killing test...\n")
            os.kill(os.getpid(), signal.SIGTERM)



class TestWGet(unittest.TestCase):

    def setUp(self):
        self.baseurl = "http://%s:%s/" % (TEST_HOST, TEST_PORT)
        self.wget = wget.WGet()

    def tearDown(self):
        self.wget = None


    def test_10_get_request(self):
        url = self.baseurl + '?a=6&b=8#x'

        # Get request: 
        self.wget.request(url, headers={'test1': 'value1', 'h2':'val2'})
        self.assertEqual(self.wget._content, None)
        self.assertEqual(self.wget._soup, None)

        self.assertEqual( self.wget.format, 'html' )

        content = self.wget.content
        self.assertTrue(re.search(r"<test>AAA&nbsp;BBB</test>\n", content))
        self.assertTrue(re.search(r"<method>'GET'</method>\n", content))
        self.assertEqual(self.wget._content, content)
        self.assertEqual(self.wget._soup, None)

        self.assertEqual(len(self.wget.select('test')), 1)
        self.assertTrue(isinstance(self.wget._soup, BeautifulSoup))
        self.assertEqual(self.wget.select('test')[0].name, 'test')
        self.assertEqual(self.wget.select('test')[0].text, u"AAA\xa0BBB")

        self.assertEqual(self.wget.select_first('path').text,
                            "'/?a=6&b=8#x'")
        self.assertEqual(self.wget.select_first('header[name=user-agent]').text,
                            "'Mozilla/4.0 (compatible; MSIE 5.5; Windows NT)'")
        self.assertEqual(self.wget.select_first('header[name=test1]').text,
                            "'value1'")
        self.assertEqual(self.wget.select_first('header[name=h2]').text,
                            "'val2'")

        self.assertEqual(len(self.wget.select('data')), 0)

        # Next request removes cached data 
        self.wget.request(url)
        self.assertEqual(self.wget._content, None)
        self.assertEqual(self.wget._soup, None)

        
    def test_20_post_request(self):
        url = self.baseurl + '?a=6'

        self.wget.request(url, data={'ax': 241, 'sa-a': '325x'},
                                headers={'test1': 'value1'})
        self.assertEqual(self.wget._content, None)
        self.assertEqual(self.wget._soup, None)

        content = self.wget.content
        self.assertTrue(re.search(r"<method>'POST'</method>\n", content))
        self.assertEqual(self.wget._content, content)
        self.assertEqual(self.wget._soup, None)

        self.assertEqual(len(self.wget.select('method')), 1)
        self.assertTrue(isinstance(self.wget._soup, BeautifulSoup))
        self.assertEqual(self.wget.select_first('method').name, 'method')
        self.assertEqual(self.wget.select_first('method').text, "'POST'")

        self.assertEqual(self.wget.select_first('path').text, "'/?a=6'")
        self.assertEqual(self.wget.select_first('header[name=user-agent]').text,
                            "'Mozilla/4.0 (compatible; MSIE 5.5; Windows NT)'")
        self.assertEqual(self.wget.select_first('header[name=test1]').text,
                            "'value1'")
        self.assertEqual(self.wget.select('header[name=h2]'), [])

        self.assertEqual(len(self.wget.select('data')), 2)
        self.assertEqual(self.wget.select('data[name=ax]')[0].text, "'241'")
        self.assertEqual(self.wget.select('data[name=sa-a]')[0].text, "'325x'")

        
    def test_20_cookies(self):
        """ Test automatic cookies handling """
        self.wget.request(self.baseurl)
        self.assertEqual(len(self.wget.select('header[name=cookie]')), 0)

        self.wget.request(self.baseurl)
        self.assertEqual(len(self.wget.select('header[name=cookie]')), 1)
        self.assertEqual(self.wget.select('header[name=cookie]')[0].text,
                            "'cookieName=cookieValue'")



if __name__ == '__main__':
    test_server = MockServer()
    test_server.run()
    try:
        unittest.main()
    finally:
        test_server.quit()

