
import unittest
from BeautifulSoup import BeautifulSoup, Tag
from SoupSelector import select, select_first, filter_to_callable, \
                            update_filters
import re


HTML = """
<html>
<head>
  <title>Page Title</title>
  <meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">
</head>
<body>
  <div id="head1" class="layout blue header">
  </div>
  <div id="content1" class="layout blue content"
    <div id="test-123" class="layout">
        <p id="pp11"><b>Test:</b> uux</p>
        <p id="pp22">par2</p>
    </div>
    <div id="empty1"></div>

    <select id="sel1" name="select-A" color="black">
        <option id="optA1" value="en">English</option>
        <optgroup id="oG1">
          <option id="optA2" value="en-US" selected>English US</option>
          <option id="optA3" value="env">ENV</option>
        </optgroup>
    </select>

    <select id="sel2" name="select-B" color="green">
        <option id="optB1" value="mytest">T1</option>
        <option id="optB2" value="testSomething[1]" selected="selected"></option>
        <option id="optB3" value="run test 2">T2</option>
    </select>

    Some text

    <table id="tab1">
        <tr id="row1">
            <td>1</td>
            <td>1</td>
            <td>1</td>
        </tr>
    </table>
    <span id="s-1" my-color="black" class="my-span layout">X</span>
  </div>
  <div id="foot1" class="footer layout" color="green">
    <span id="specL" class="special-layout">spec</span>
  </div>
</body>
</html>
"""


class Test_SoupSelector(unittest.TestCase):
    """ Unit tests for SoupSelector module """

    def setUp(self):
        self.soup = BeautifulSoup(HTML)


    def assertSelect(self, selector, expected_tokens, *testName):
        result = select(self.soup, selector)
        self.assertEqual(
                [item.get('id') for item in result],
                expected_tokens,
                *testName
            )

    def test_filter_to_callable(self):
        filter_def = lambda x: x ** 3
        call = filter_to_callable( filter_def )
        self.assertEqual( filter_def, call)
        self.assertEqual( filter_def(4), call(4) )

        call = filter_to_callable( True )
        self.assertTrue( callable(call) )
        self.assertFalse( call(None) )
        self.assertTrue( call('') )
        self.assertTrue( call('123') )

        call = filter_to_callable( re.compile('\w{3}') )
        self.assertTrue( callable(call) )
        self.assertFalse( call(None) )
        self.assertFalse( call('a') )
        self.assertFalse( call('a1-b2') )
        self.assertTrue( call('%a1b2') )

        call = filter_to_callable( ['aC', 'xax', ''] )
        self.assertTrue( callable(call) )
        self.assertFalse( call(None) )
        self.assertTrue( call('') )
        self.assertFalse( call('ax') )
        self.assertTrue( call('xax') )
        self.assertTrue( call('aC') )

        call = filter_to_callable( 'test1' )
        self.assertTrue( callable(call) )
        self.assertFalse( call(None) )
        self.assertFalse( call('') )
        self.assertFalse( call('test') )
        self.assertFalse( call('test123') )
        self.assertTrue( call('test1') )
        self.assertTrue( call(u'test1') )

        call = filter_to_callable( u'test1' )
        self.assertTrue( callable(call) )
        self.assertFalse( call(None) )
        self.assertFalse( call('test') )
        self.assertTrue( call('test1') )
        self.assertTrue( call(u'test1') )


    def test_update_filters(self):
        f = {
            'name': 'div',
            'attrs': {
                'class': 'ok',
                'value': re.compile('\w{4}')
            }
        }
        update_filters(f, new_key=4126, attrs={
                'value': ['Axa', 'kl4-0as', '18799', 'AAAA'],
                'color': 'black'
            })
        self.assertEqual(sorted(f.keys()), ['attrs', 'name', 'new_key'])
        self.assertEqual(f['name'], 'div')
        self.assertEqual(f['new_key'], 4126)

        self.assertEqual(sorted(f['attrs'].keys()),
                        ['class', 'color', 'value'])
        self.assertEqual(f['attrs']['class'], 'ok')
        self.assertEqual(f['attrs']['color'], 'black')

        self.assertTrue( callable( f['attrs']['value'] ) )
        self.assertFalse( f['attrs']['value'](None) )
        self.assertFalse( f['attrs']['value']('Axa') )
        self.assertFalse( f['attrs']['value']('kl4-0as') )
        self.assertTrue( f['attrs']['value']('18799') )
        self.assertTrue( f['attrs']['value']('AAAA') )
        self.assertFalse( f['attrs']['value']('AAAAA') )


    def test_tag(self):
        self.assertSelect("p", ["pp11", "pp22"])
        self.assertSelect("div",
                ["head1", "content1", "test-123", "empty1", "foot1"])
        self.assertSelect("unknown", [])


    def test_class(self):
        self.assertSelect(".layout",
                ["head1", "content1", "test-123", "s-1", "foot1"])
        self.assertSelect(".footer", ["foot1"])
        self.assertSelect(".foot", [])


    def test_id(self):
        self.assertSelect("#foot1", ["foot1"])
        self.assertSelect("#test-123", ["test-123"])
        self.assertSelect("#test-1", [])
        

    def test_attr_exists(self):
        self.assertSelect("[name]", ["sel1", "sel2"])
        self.assertSelect("[selected]", ["optA2", "optB2"])
        self.assertSelect("[my-color]", ["s-1"])
        self.assertSelect("[sel2]", [])


    def test_attr_equals(self):
        self.assertSelect('[my-color=black]', ["s-1"])
        self.assertSelect('[name="select-B"]', ["sel2"])
        self.assertSelect('[value=run test 2]', ["optB3"])
        self.assertSelect('[value="run test 2"]', ["optB3"])
        self.assertSelect('[value="testSomething[1]"]', ["optB2"])
        self.assertSelect('[class=layout]', ["test-123"])
        self.assertSelect('[value="test"]', [])


    def test_attr_contains_word(self):
        self.assertSelect('[class~=layout]',
                ["head1", "content1", "test-123", "s-1", "foot1"])
        self.assertSelect('[value~="test"]', ["optB3"])
        self.assertSelect('[value~=123]', [])


    def test_attr_startswith(self):
        self.assertSelect('[name^=sele]', ["sel1", "sel2"])
        self.assertSelect('[value^=en]', ["optA1", "optA2", "optA3"])
        self.assertSelect('[value^="test"]', ["optB2"])
        self.assertSelect('[class^=layout]', ["head1", "content1", "test-123"])
        self.assertSelect('[value^="testing"]', [])


    def test_attr_endswith(self):
        self.assertSelect('[class$=layout]',
                ["test-123", "s-1", "foot1", "specL"])
        self.assertSelect('[id$=123]', ["test-123"])
        self.assertSelect('[value$="test"]', ["optB1"])
        self.assertSelect('[value$="my"]', [])


    def test_attr_contains(self):
        self.assertSelect('[class*=layout]',
                ["head1", "content1", "test-123", "s-1", "foot1", "specL"])
        self.assertSelect('[value*="test"]', ["optB1", "optB2", "optB3"])
        self.assertSelect('[value*="run t"]', ["optB3"])
        self.assertSelect('[value*="opt"]', [])


    def test_attr_lang_prefix(self):
        self.assertSelect('[class|=layout]', ["test-123"])
        self.assertSelect('[value|="en"]', ["optA1", "optA2"])
        self.assertSelect('[name|="select"]', ["sel1", "sel2"])
        self.assertSelect('[value|="run"]', [])


    def test_combine_attrs(self):
        self.assertSelect('[value^=en][selected]', ["optA2"])
        self.assertSelect('[selected][value^=en]', ["optA2"])
        self.assertSelect('[color=green]', ["sel2", "foot1"])
        self.assertSelect('[color=green][name|="select"]', ["sel2"])
        self.assertSelect('[value*=test][value^=run]', ["optB3"])
        self.assertSelect('[value^=run][value*=test]', ["optB3"])
        self.assertSelect('[value^=run][value$=test]', [])
        self.assertSelect('[color="green"][selected]', [])


    def test_combine_2(self):
        self.assertSelect("div.layout",
                ["head1", "content1", "test-123", "foot1"])
        self.assertSelect("div.layout.blue", ["head1", "content1"])
        self.assertSelect("div[class*=blue].layout", ["head1", "content1"])
        self.assertSelect("div[class~=blue].layout", ["head1", "content1"])
        self.assertSelect("div.blue[color=green]", [])
        self.assertSelect("div.blue#content1.layout", ["content1"])
        self.assertSelect("div[class=blue].layout", [])
        self.assertSelect("*[class*=layout]",
                ["head1", "content1", "test-123", "s-1", "foot1", "specL"])
        self.assertSelect(".layout[color=green]", ["foot1"])
        self.assertSelect("*.layout[color=green]", ["foot1"])
        self.assertSelect("option[selected]", ["optA2", "optB2"])
        self.assertSelect("option[selected][value=en-US]", ["optA2"])
        self.assertSelect("option[selected][value=en]", [])


    def test_descendant(self):
        self.assertSelect('select option[selected]', ['optA2', 'optB2'])
        self.assertSelect('div div', ['test-123', 'empty1'])
        self.assertSelect('select  *  option', ['optA2', 'optA3'])
        self.assertSelect('div body', [])
        self.assertSelect('body div#content1 option[selected]', ['optA2', 'optB2'])
        self.assertSelect('body div#content1 * option[selected]', ['optA2', 'optB2'])
        self.assertSelect('select select', [])


    def test_child(self):
        self.assertSelect('select > option[selected]', ['optB2'])
        self.assertSelect('body > div', ['head1', 'content1', 'foot1'])
        self.assertSelect('select > option', ['optA1', 'optB1', 'optB2', 'optB3'])
        self.assertSelect('div > body', [])
        self.assertSelect('body div#content1 > option[selected]', [])
        self.assertSelect('select > select', [])


    def test_adjacent_sibling(self):
        self.assertSelect('#head1 + #content1', ['content1'])
        self.assertSelect('#head1 + div', ['content1'])
        self.assertSelect('#head1 + #foot1', [])
        self.assertSelect('select + table', ['tab1'])
        self.assertSelect('option + option', ['optA3', 'optB2', 'optB3'])
        self.assertSelect('table + select', [])


    def test_general_sibling(self):
        self.assertSelect('#head1 ~ #content1', ['content1'])
        self.assertSelect('#head1 ~ div', ['content1', 'foot1'])
        self.assertSelect('#head1 ~ #foot1', ['foot1'])
        self.assertSelect('select#sel1 ~ span', ['s-1'])
        self.assertSelect('option ~ option', ['optA3', 'optB2', 'optB3'])
        self.assertSelect('table ~ select', [])


    def test_multiple_same_items(self):
        result = select(self.soup, "tr#row1 td")
        self.assertEqual([t.name for t in result], ['td', 'td', 'td'])
        

    def test_select_limit(self):
        result = self.soup.select("body div", limit=4)
        self.assertEqual(
                [item.get('id') for item in result],
                ["head1", "content1", "test-123", "empty1"]
            )

        result = self.soup.select("body div", limit=2)
        self.assertEqual(
                [item.get('id') for item in result],
                ["head1", "content1"]
            )

        result = self.soup.select("body div", limit=1)
        self.assertEqual(
                [item.get('id') for item in result],
                ["head1"]
            )

        
    def test_select_first(self):
        result = select_first(self.soup, '#foot1')
        self.assertTrue(isinstance(result, Tag))
        self.assertEqual(result.get('id'), 'foot1')

        result = select_first(self.soup, 'div')
        self.assertTrue(isinstance(result, Tag))
        self.assertEqual(result.get('id'), 'head1')

        result = select_first(self.soup, 'aha')
        self.assertEqual(result, None)


    def test_BeautifulSoup_extension(self):
        result = self.soup.select("body div")
        self.assertEqual(
                [item.get('id') for item in result],
                ["head1", "content1", "test-123", "empty1", "foot1"]
            )

        item = self.soup.select_first("body div")
        self.assertTrue(isinstance(item, Tag))
        self.assertEqual(item.get('id'), 'head1')

        result = result.select("div")
        self.assertEqual(
                [item.get('id') for item in result],
                ["test-123", "empty1"]
            )

        result = result.select("p")
        self.assertEqual(
                [item.get('id') for item in result],
                ["pp11", "pp22"]
            )

        item = self.soup.select_first("p")
        self.assertTrue(isinstance(item, Tag))
        self.assertEqual(item.get('id'), 'pp11')

        result = result.select("p.x")
        self.assertEqual(
                [item.get('id') for item in result],
                []
            )

        item = self.soup.select_first("p.x")
        self.assertEqual(item, None)


if __name__ == '__main__':
    unittest.main()

