# -*- encoding: utf-8 -*-
""" StreamCZ plugin module for www.stream.cz """

from basePlugin import *
import re
import urllib
import urlparse


CDN_URL  = "http://cdn-dispatcher.stream.cz/?id=%d"
RE_URL  = re.compile('''url\(['"]([^'")]+)['"]\)''')
RE_CDN_ID_HQ = re.compile('cdnHQ=(\d+)')
RE_CDN_ID_LQ = re.compile('cdnLQ=(\d+)')
RE_CDN_ID_OLD = re.compile('cdnID=(\d+)')
RE_PAGE = re.compile('&page=(\d+)')

LIST_RE_CDN_ID = (
    ( RE_CDN_ID_HQ, RE_CDN_ID_OLD, RE_CDN_ID_LQ ),
    ( RE_CDN_ID_LQ, RE_CDN_ID_OLD, RE_CDN_ID_HQ ),
)

class StreamCZPlugin(BasePlugin):
    LOGPREFIX = "[StreamCZ plugin] "
    BASE_URL = "http://www.stream.cz"
    SEARCH_ORDER = ['relevance', 'timecreate', 'starcount']
    MAX_COUNT = [10, 20, 30, 40, 50]

    @plugin_folder
    def root(self):
        """ plugin root folder """
        self.core.setSorting('NONE')
        yield PFolder(u'Hledat video', call='search')
        yield PFolder(u'Televize', call='list_televize')
        yield PFolder(u'Uživatelská videa', call='list_categories',
                        param={'path': '/kategorie/2-uzivatelska-videa'})
        yield PFolder(u'Hudba', call='list_hudba')
        yield PFolder(u'Témata', call='list_categories',
                        param={'path': '/tema'})


    @plugin_folder
    def list_televize(self):
        self.wget.request( self.BASE_URL + '/televize' )
        for item in self.wget.select('ul.uniMenu > li'):
            item_link = item.select_first('a')
            yield PFolder(item_link.text, call='list_tv_channel',
                            param={'path': item_link['href']})


    @plugin_folder
    def list_tv_channel(self, path):
        url = self.full_url(path)
        self.wget.request(url)

        subitems = self.wget.select('ul.uniMenu ul.sub li')
        if not subitems:
            for item in self.list_videos_on_page():
                yield item
            return

        yield PFolder(u'Vše', call='list_videos', param={'path': path})

        for item in subitems:
            item_link = item.select_first('a')
            yield PFolder(item_link.text, call='list_videos',
                            param={'path': item_link['href']})


    @plugin_folder
    def list_hudba(self):
        yield PFolder(u'TOP 20', call='list_videos',
                    param={'path': 'http://music.stream.cz/klipy/top20'})
        yield PFolder(u'Nejnovější klipy', call='list_videos',
                    param={'path': 'http://music.stream.cz/klipy/nejnovejsi'})
        yield PFolder(u'Seznam interpretů', call='list_hudba_interprets')
        yield PFolder(u'Hudební videoarchiv', call='list_videos',
                    param={'path': 'http://music.stream.cz/porady/hudebni-videoarchiv'})


    @plugin_folder
    def list_hudba_interprets(self):
        self.wget.request('http://music.stream.cz/seznam-interpretu')
        for a_tag in self.wget.select('.all_interpreter_list a'):
            yield PFolder(a_tag.text,
                    call = 'list_videos',
                    param = {'path': self.full_link_to(a_tag['href'])}
                )


    @plugin_folder
    def list_categories(self, path):
        url = self.full_url(path)
        self.wget.request(url)
        for item in self.wget.select('ul.uniMenu > li'):
            item_link = item.select_first('a')
            yield PFolder(item_link.text, call='list_videos',
                            param={'path': item_link['href']})
    

    @plugin_folder
    def list_videos(self, path):
        """ list videos """
        url = self.full_url(path)
        self.wget.request( url )

        for item in self.list_videos_on_page():
            yield item


    ### Search ###

    @plugin_folder
    def search(self):
        """ search folder """
        yield PFolder(label=u'<Nové hledání>', call='new_search')
        for qtext in self.get_saved_searches():
            del_url = self.core.generateURL(call='delete_search', qtext=qtext)
            del_action = 'XBMC.RunPlugin(%s)' % del_url
            yield PFolder(label=qtext, call='search_result',
                            param={'qtext': qtext},
                            menuItems=[('Vymazat', del_action)] )


    @plugin_call
    def new_search(self):
        """ search new text """
        qtext = self.core.textDialog('Search videos')
        if not qtext:
            raise CancelOperation()
        list_url = self.core.generateURL(call='search_result', qtext=qtext)
        self.core.execute('Container.Update(%s)' % list_url)


    @plugin_folder
    def search_result(self, qtext, page=0):
        """ Search items on web and display result in as plugin folder """
        order = self.SEARCH_ORDER[ int(self.core.getSetting('search_order')) ]
        page  = int(page)
        url = '%s/?a=search&search_text=%s&resultsorting=%s&page=%d' \
                % (self.BASE_URL, urllib.quote_plus(qtext), order, page)
        self.wget.request(url)

        found = self.wget.select('div.videoList')
        if not found:
            self.core.okDialog('StreamCZ', u'Žádné video nenalezeno.')
            raise CancelOperation()

        self.save_search(qtext)

        # add paging links
        for pgLink in self.wget.select('.paging a'):
            m = RE_PAGE.search( pgLink.get('href', '') )
            if m and int(m.group(1)) > page:
                yield PFolder(label=u'<Další stránka>', call='search_result',
                                param={'qtext': qtext, 'page': m.group(1)}
                            )
                break


        for item in found:
            info = {'plot': item.select_first('div.videoListData p').text}

            iconURL=None
            m = RE_URL.search( item.select_first('a.videoListImg').get('style', '') )
            if m:
                iconURL = m.group(1)

            title_link = item.select_first('h5 a')
            yield PItem(label=title_link.text,
                        call='play',
                        param={'path': title_link['href']},
                        info=info,
                        iconURL=iconURL
                    )



    @plugin_call
    def delete_search(self, qtext):
        """ Delete search from saved search """
        saved_list = self.get_saved_searches()
        if qtext in saved_list:
            saved_list.remove(qtext)
            self.set_saved_searches(saved_list)
        self.core.execute( "Container.Refresh" )


    ### PLAY ###

    @plugin_call
    def play(self, path):
        """ Play video """
        url = self.full_url( path )
        self.core.newProgress('StreamCZ', u'Hledám video stream')
        self.core.updateProgress(30)
        self.debug('play url: '+repr(url))
        self.wget.request( url )

        body = self.wget.select_first('body')
        title = body.select_first('#playerContainer h1') \
                or body.select_first('.col_main_b h1') \
                or body.select_first('#player_detail h1') \
                or body.select_first('#detail_player h1') \
                or body.select_first('h1')
        plot  = body.select_first('#videoDescription') \
                or body.select_first('#video_popisek') \
                or body.select_first('#player_detail_popisek') \
                or body.select_first('.player_detail_text')

        info = {}
        if plot and plot.text:
            info['plot'] = plot.text
        item = PItem(
                    label=title.text,
                    info=info,
                )
        player = body.select_first('#player')
        if not player:
            raise Exception("#player not found! - url: " + url)
        cdn_id = self.get_cdn_id( unicode(player) )
        if not cdn_id:
            raise Exception("cdn_id not found!")
        stream_url = CDN_URL % cdn_id
        self.debug('streamURL: '+repr(stream_url) )
        self.core.closeProgress()

        self.core.play(stream_url, item)


    ### Internal methods ###


    def get_cdn_id(self, content):
        for regexp in LIST_RE_CDN_ID[int(self.core.getSetting("video_quality"))]:
            m = regexp.search(content)
            if m:
                return int( m.group(1) )
        return None
        

    def get_saved_searches(self):
        """ Get saved searches list """
        try:
            items = self.core.getSetting("saved_list").split('\n')
            return filter(None, items)
        except:
            return []


    def set_saved_searches(self, saved_list):
        """ Set saved searches list """
        self.core.setSetting('saved_list', '\n'.join(saved_list))


    def save_search(self, qtext):
        """ Save search to saved list """
        saved_list = self.get_saved_searches()
        if qtext not in saved_list:
            saved_list.insert(0, qtext)
            try:
                maxCount = self.MAX_COUNT[int(self.core.getSetting("max_saved_count"))]
            except:
                maxCount = 10
            if len(saved_list) > maxCount:
                saved_list = saved_list[:maxCount]
            self.set_saved_searches(saved_list)


    def full_url(self, path):
        if '://' not in path:
            return self.BASE_URL + path
        return path


    def list_videos_on_page(self):
        channels = self.wget.select('#sekce_kanalu li a')
        if channels:
            for a in channels:
                yield PFolder(a.text, call='list',
                            param={'path': a['href']}
                        )

        items = self.wget.select('div.matrixThreeVideoList') \
                    or self.wget.select('div.videoListContent div.videoList') \
                    or self.wget.select('div.videolist_one') \
                    or self.wget.select('div.kanal_1video') \
                    or self.wget.select('ul.item_list li')
        for item_tag in items:
            plot_div = item_tag.select_first('p') \
                        or item_tag.select_first('div.videolist_one_text') \
                        or item_tag.select_first('div.kanal_1video_text') \
                        or item_tag.select_first('a.interpreter')
            title_a = item_tag.select_first('h5 a') \
                        or item_tag.select_first('div.videolist_one_content a') \
                        or item_tag.select_first('a.kanal_1video_title') \
                        or item_tag.select_first('a.title')
            item = PItem(label = title_a.text,
                        call = 'play',
                        param = {'path': self.full_link_to(title_a['href'])},
                        info = {'plot': plot_div.text}
                    )
            img_tag = item_tag.select_first('a img.thumb')
            if img_tag and img_tag['src']:
                item.iconURL = img_tag['src']
            else:
                img_a = item_tag.select_first('a.videoListImg') \
                            or item_tag.select_first('a.videolist_one_pic') \
                            or item_tag.select_first('a.kanal_1video_pic')
                m = RE_URL.search( img_a.get('style', '') )
                if m:
                    item.iconURL = m.group(1)
            yield item

        nextPageLinkPath = self.next_page_link()
        if nextPageLinkPath:
            yield PFolder(u'<Další strana>', 
                        call='list_videos',
                        param={'path': self.full_link_to(nextPageLinkPath)}
                    )


    def next_page_link(self):
        paging_links = self.wget.select('div.paging a') \
                        or self.wget.select('div.pager a')
        for a in paging_links:
            if re.search(u'další|následující', a.text, re.U):
                return a['href']
        return None


    def full_link_to(self, path):
        if '://' in path:
            return path
        return urlparse.urljoin(self.wget.request_url, path)

